/* 
 *  MonitoredUserData.java
 * 
 *  Copyright 2016 Avaya Inc. All Rights Reserved.
 * 
 *  Usage of this source is bound to the terms described
 *  in AvayaLicenseSDK.rtf.
 * 
 *  Avaya - Confidential & Proprietary. Use pursuant to your signed agreement
 *  or Avaya Policy
 * 
 */
package com.avaya.ccs.javafxrefclient;

import com.avaya.ccs.api.CodeListI;
import com.avaya.ccs.api.ContactCenterCodeI;
import com.avaya.ccs.api.MonitoredUserI;
import com.avaya.ccs.api.enums.ContactType;
import com.avaya.ccs.api.enums.UserState;
import com.avaya.ccs.api.exceptions.ObjectInvalidException;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import javafx.util.Pair;

public class MonitoredUserData extends Data {

    // Properties
    private String id = "0";
    private List<ContactType> contactTypes = new ArrayList<>();
    private ContactCenterCodeI code;
    private String firstName = "";
    private String lastName = "";
    private String fullName = "";
    private String loginId = "";
    private UserState state = UserState.Unknown;
    private String supervisorId = "";

    // Capabilities
    private boolean canForceLogout = false;
    private boolean canForceNotReady = false;
    private boolean canForceReady = false;

    private CodeListI codes = null;

    // Table specific properties
    private boolean isSelected = false;

    private static final Logger LOG = Logger.getLogger(MonitoredUserData.class);

    public MonitoredUserData(MonitoredUserI user) {
        String methodName = "MonitoredUserData() ";
        if (user != null) {
            try {
                // Properties
                this.id = user.getId();
                this.contactTypes = user.getContactTypes();
                this.code = user.getCode();
                this.firstName = user.getFirstName();
                this.lastName = user.getLastName();
                this.fullName = user.getFullName();
                this.loginId = user.getLoginId();
                this.state = user.getState();
                this.supervisorId = user.getSupervisorId();

                // Capabilities
                this.canForceLogout = user.canForceLogout();
                this.canForceNotReady = user.canForceNotReady();
                this.canForceReady = user.canForceReady();
                this.isSelected = false;

            } catch (ObjectInvalidException ex) {
                LOG.error(methodName, "ObjectInvalidException while accessing MonitoredUserData", ex);
                // Set all properties to default values
                this.id = "0";
                this.contactTypes = new ArrayList<>();
                this.code = null;
                this.firstName = "";
                this.lastName = "";
                this.fullName = "";
                this.loginId = "";
                this.state = UserState.Unknown;
                this.supervisorId = "";
                // Set all capabilities to default values
                this.canForceLogout = false;
                this.canForceNotReady = false;
                this.canForceReady = false;
                this.isSelected = false;
            }
        }
    }

    public String toString() {
        String str = "ID:" + id;
        // Properties
        str += " contactTypes:" + contactTypes + " code:" + code + " firstName:" + firstName + " lastName:" + lastName
                + " fullName:" + fullName + " loginId:" + loginId + " state:" + state.name() + " supervisorId:"
                + supervisorId;
        // Capabilities
        str += " canForceLogout:" + canForceLogout + " canForceNotReady:" + canForceNotReady + " canForceReady:"
                + canForceReady;
        return str;
    }

    public MonitoredUserData(String id, boolean deleted) {
        this.id = id;
        this.setDeleted(deleted);
    }

    @Override
    public List<Pair<String, String>> getProperties() {
        List<Pair<String, String>> properties = new LinkedList<>();
        properties.add(new Pair<String, String>("ID", this.getId()));
        properties.add(new Pair<String, String>("Contact Types", typesToString()));
        properties.add(new Pair<String, String>("Code", codeToString()));
        properties.add(new Pair<String, String>("First Name", this.getFirstName()));
        properties.add(new Pair<String, String>("Last Name", this.getLastName()));
        properties.add(new Pair<String, String>("Full Name", this.getFullName()));
        properties.add(new Pair<String, String>("Login ID", this.getLoginId()));
        properties.add(new Pair<String, String>("State", this.stateToString()));
        properties.add(new Pair<String, String>("Supervisor ID", this.getSupervisorId()));

        return properties;
    }

    @Override
    public List<Pair<String, Boolean>> getCaps() {
        List<Pair<String, Boolean>> caps = new LinkedList<>();
        caps.add(new Pair<String, Boolean>("CanForceLogout", this.canForceLogout()));
        caps.add(new Pair<String, Boolean>("CanForceNotReady", this.canForceNotReady()));
        caps.add(new Pair<String, Boolean>("CanForceReady", this.canForceReady()));
        return caps;
    }

    public boolean isSelected() {
        return isSelected;
    }

    public void setIsSelected(boolean isSelected) {
        this.isSelected = isSelected;
    }

    private String typesToString() {
        if (getContactTypes().isEmpty()) {
            return "NONE";
        }

        String types = "";
        for (ContactType type : getContactTypes()) {
            types += type.name() + " ";
        }
        return types;
    }

    private String codeToString() {
        if (this.getCode() == null) {
            return "<NULL>";
        } else {
            return getCode().getName() + " " + getCode().getNumber();
        }
    }

    private String stateToString() {
        if (this.state == null) {
            return "<NULL>";
        } else {
            return this.state.name();
        }
    }

    public UserState getState() {
        return this.state;
    }

    @Override
    public String getId() {
        return id;
    }

    public List<ContactType> getContactTypes() {
        return contactTypes;
    }

    public ContactCenterCodeI getCode() {
        return code;
    }

    public String getFirstName() {
        return firstName;
    }

    public String getLastName() {
        return lastName;
    }

    public String getFullName() {
        return fullName;
    }

    public String getLoginId() {
        return loginId;
    }

    public String getSupervisorId() {
        return supervisorId;
    }

    public boolean canForceLogout() {
        return canForceLogout;
    }

    public boolean canForceNotReady() {
        return canForceNotReady;
    }

    public boolean canForceReady() {
        return canForceReady;
    }

    public CodeListI getCodes() {
        return codes;
    }

    public void setCodes(CodeListI codes) {
        this.codes = codes;
    }

    @Override
    public void update(Data newData) {
        MonitoredUserData newMonitoredUserData = (MonitoredUserData) newData;
        // Properties
        this.id = newMonitoredUserData.getId();
        this.contactTypes = newMonitoredUserData.getContactTypes();
        this.code = newMonitoredUserData.getCode();
        this.firstName = newMonitoredUserData.getFirstName();
        this.lastName = newMonitoredUserData.getLastName();
        this.fullName = newMonitoredUserData.getFullName();
        this.loginId = newMonitoredUserData.getLoginId();
        this.state = newMonitoredUserData.getState();
        this.supervisorId = newMonitoredUserData.getSupervisorId();

        // Capabilities
        this.canForceLogout = newMonitoredUserData.canForceLogout();
        this.canForceNotReady = newMonitoredUserData.canForceNotReady();
        this.canForceReady = newMonitoredUserData.canForceReady();

        this.codes = newMonitoredUserData.getCodes();

        // Table specific properties
        this.isSelected = newMonitoredUserData.isSelected();
    }
}
